<?php
// Authentication endpoints

// RESTful yapı: action path'ten alınıyor (/api/v1/auth/login)
// Eğer path'te action yoksa, GET parametresinden al (geriye dönük uyumluluk)
$action = $action ?? $_GET['action'] ?? '';

switch ($request_method) {
    case 'POST':
        // POST için varsayılan action login
        if (empty($action)) {
            $action = 'login';
        }
        
        switch ($action) {
            case 'login':
                // Parent login
                $input = json_decode(file_get_contents('php://input'), true);
                
                if (!$input || !isset($input['username']) || !isset($input['password'])) {
                    api_error('Kullanıcı adı ve şifre gerekli', 400);
                }
                
                $username = trim($input['username']);
                $password = trim($input['password']);
                
                // Önce users tablosundan kullanıcıyı bul (veli veya parent)
                $stmt = $pdo->prepare("
                    SELECT u.id as user_id, u.username, u.password, u.email, u.full_name, u.phone, u.user_type
                    FROM users u
                    WHERE u.username = ? AND (u.user_type = 'veli' OR u.user_type = 'parent')
                ");
                $stmt->execute([$username]);
                $user = $stmt->fetch();
                
                if (!$user || !password_verify($password, $user['password'])) {
                    api_error('Geçersiz kullanıcı adı veya şifre', 401);
                }
                
                // Parents tablosunda kayıt var mı kontrol et
                $parent_id = null;
                try {
                    $stmt = $pdo->prepare("SELECT id FROM parents WHERE user_id = ?");
                    $stmt->execute([$user['user_id']]);
                    $parent_record = $stmt->fetch();
                    
                    if ($parent_record) {
                        // Parents tablosunda kayıt var
                        $parent_id = $parent_record['id'];
                    } else {
                        // Parents tablosunda kayıt yok, oluştur
                        $stmt = $pdo->prepare("
                            INSERT INTO parents (user_id, phone, address, created_at) 
                            VALUES (?, ?, '', NOW())
                        ");
                        $stmt->execute([$user['user_id'], $user['phone'] ?? '']);
                        $parent_id = $pdo->lastInsertId();
                    }
                } catch (PDOException $e) {
                    // Parents tablosu yoksa veya hata varsa, parent_id'yi user_id'ye eşitle
                    // (Eski sistem uyumluluğu için)
                    $parent_id = $user['user_id'];
                }
                
                // JWT token oluştur
                $payload = [
                    'user_id' => $user['user_id'],
                    'parent_id' => $parent_id,
                    'username' => $user['username'],
                    'full_name' => $user['full_name'],
                    'iat' => time(),
                    'exp' => time() + (24 * 60 * 60) // 24 saat
                ];
                
                $token = JWT::encode($payload);
                
                // Son giriş tarihini güncelle
                $stmt = $pdo->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
                $stmt->execute([$user['user_id']]);
                
                api_response([
                    'token' => $token,
                    'user' => [
                        'id' => $parent_id,
                        'username' => $user['username'],
                        'full_name' => $user['full_name'],
                        'email' => $user['email'],
                        'phone' => $user['phone']
                    ]
                ], 200, 'Giriş başarılı');
                break;
                
            case 'register':
                // Parent kayıt (isteğe bağlı)
                $input = json_decode(file_get_contents('php://input'), true);
                
                if (!$input || !isset($input['username']) || !isset($input['password']) || !isset($input['full_name'])) {
                    api_error('Kullanıcı adı, şifre ve ad soyad gerekli', 400);
                }
                
                $username = trim($input['username']);
                $password = trim($input['password']);
                $full_name = trim($input['full_name']);
                $email = trim($input['email'] ?? '');
                $phone = trim($input['phone'] ?? '');
                
                // Kullanıcı adı kontrolü
                $stmt = $pdo->prepare("SELECT id FROM users WHERE username = ?");
                $stmt->execute([$username]);
                if ($stmt->fetch()) {
                    api_error('Bu kullanıcı adı zaten kullanılıyor', 400);
                }
                
                // Kullanıcı oluştur
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                $stmt = $pdo->prepare("
                    INSERT INTO users (username, password, email, full_name, user_type, created_at) 
                    VALUES (?, ?, ?, ?, 'parent', NOW())
                ");
                $stmt->execute([$username, $hashed_password, $email, $full_name]);
                $user_id = $pdo->lastInsertId();
                
                // Parent kaydı oluştur
                $stmt = $pdo->prepare("
                    INSERT INTO parents (user_id, phone, address, created_at) 
                    VALUES (?, ?, '', NOW())
                ");
                $stmt->execute([$user_id, $phone]);
                $parent_id = $pdo->lastInsertId();
                
                // JWT token oluştur
                $payload = [
                    'user_id' => $user_id,
                    'parent_id' => $parent_id,
                    'username' => $username,
                    'full_name' => $full_name,
                    'iat' => time(),
                    'exp' => time() + (24 * 60 * 60)
                ];
                
                $token = JWT::encode($payload);
                
                api_response([
                    'token' => $token,
                    'user' => [
                        'id' => $parent_id,
                        'username' => $username,
                        'full_name' => $full_name,
                        'email' => $email,
                        'phone' => $phone
                    ]
                ], 201, 'Kayıt başarılı');
                break;
                
            case 'refresh':
                // Token yenileme
                $payload = authenticate();
                
                // Yeni token oluştur
                $new_payload = [
                    'user_id' => $payload['user_id'],
                    'parent_id' => $payload['parent_id'],
                    'username' => $payload['username'],
                    'full_name' => $payload['full_name'],
                    'iat' => time(),
                    'exp' => time() + (24 * 60 * 60)
                ];
                
                $token = JWT::encode($new_payload);
                
                api_response([
                    'token' => $token
                ], 200, 'Token yenilendi');
                break;
                
            default:
                api_error('Geçersiz işlem', 400);
        }
        break;
        
    case 'GET':
        // GET için varsayılan action verify
        if (empty($action)) {
            $action = 'verify';
        }
        
        switch ($action) {
            case 'verify':
                // Token doğrulama
                $payload = authenticate();
                
                // Kullanıcı bilgilerini al (parents tablosu yoksa users'dan al)
                try {
                    $stmt = $pdo->prepare("
                        SELECT p.id, p.phone, p.address, u.username, u.email, u.full_name
                        FROM parents p
                        JOIN users u ON p.user_id = u.id
                        WHERE p.id = ?
                    ");
                    $stmt->execute([$payload['parent_id']]);
                    $parent = $stmt->fetch();
                    
                    if ($parent) {
                        api_response([
                            'user' => [
                                'id' => $parent['id'],
                                'username' => $parent['username'],
                                'full_name' => $parent['full_name'],
                                'email' => $parent['email'],
                                'phone' => $parent['phone']
                            ]
                        ], 200, 'Token geçerli');
                        break;
                    }
                } catch (PDOException $e) {
                    // Parents tablosu yoksa users'dan al
                }
                
                // Parents tablosu yoksa veya kayıt yoksa users'dan al
                $stmt = $pdo->prepare("
                    SELECT u.id, u.username, u.email, u.full_name, u.phone
                    FROM users u
                    WHERE u.id = ?
                ");
                $stmt->execute([$payload['user_id']]);
                $user = $stmt->fetch();
                
                if (!$user) {
                    api_error('Kullanıcı bulunamadı', 404);
                }
                
                api_response([
                    'user' => [
                        'id' => $payload['parent_id'] ?? $user['id'],
                        'username' => $user['username'],
                        'full_name' => $user['full_name'],
                        'email' => $user['email'],
                        'phone' => $user['phone']
                    ]
                ], 200, 'Token geçerli');
                break;
                
            default:
                api_error('Geçersiz işlem', 400);
        }
        break;
        
    default:
        api_error('Desteklenmeyen HTTP metodu', 405);
}
?>
