<?php
$page_title = 'Raporlar ve Analiz - BAŞKENT 1910 GÜCÜ SK';
require_once __DIR__ . '/../includes/header.php';

$report_type = $_GET['type'] ?? 'financial';
$period = $_GET['period'] ?? 'monthly';
$start_date = $_GET['start_date'] ?? date('Y-m-01');
$end_date = $_GET['end_date'] ?? date('Y-m-t');

$message = '';
$error = '';

// Rapor verilerini hazırla
$report_data = [];

if ($report_type === 'financial') {
    // Mali rapor verileri
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(transaction_date, '%Y-%m') as month,
            SUM(CASE WHEN transaction_type = 'income' THEN amount ELSE 0 END) as income,
            SUM(CASE WHEN transaction_type = 'expense' THEN amount ELSE 0 END) as expense,
            SUM(CASE WHEN transaction_type = 'income' THEN amount ELSE -amount END) as balance
        FROM cash_transactions 
        WHERE transaction_date BETWEEN ? AND ?
        GROUP BY DATE_FORMAT(transaction_date, '%Y-%m')
        ORDER BY month DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $report_data['monthly'] = $stmt->fetchAll();
    
    // Kategori bazlı gelir-gider analizi
    $stmt = $pdo->prepare("
        SELECT 
            category,
            transaction_type,
            SUM(amount) as total_amount,
            COUNT(*) as transaction_count
        FROM cash_transactions 
        WHERE transaction_date BETWEEN ? AND ?
        GROUP BY category, transaction_type
        ORDER BY total_amount DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $report_data['categories'] = $stmt->fetchAll();
    
    // Toplam istatistikler
    $stmt = $pdo->prepare("
        SELECT 
            SUM(CASE WHEN transaction_type = 'income' THEN amount ELSE 0 END) as total_income,
            SUM(CASE WHEN transaction_type = 'expense' THEN amount ELSE 0 END) as total_expense,
            COUNT(CASE WHEN transaction_type = 'income' THEN 1 END) as income_count,
            COUNT(CASE WHEN transaction_type = 'expense' THEN 1 END) as expense_count
        FROM cash_transactions 
        WHERE transaction_date BETWEEN ? AND ?
    ");
    $stmt->execute([$start_date, $end_date]);
    $report_data['totals'] = $stmt->fetch();
    
} elseif ($report_type === 'athletes') {
    // Sporcu rapor verileri
    $report_data['total_athletes'] = $pdo->query("SELECT COUNT(*) as total FROM athletes WHERE status = 'active'")->fetch()['total'];
    $report_data['new_athletes'] = $pdo->query("SELECT COUNT(*) as total FROM athletes WHERE registration_date >= DATE_SUB(NOW(), INTERVAL 30 DAY)")->fetch()['total'];
    
    // Yaş grupları
    $report_data['age_groups'] = $pdo->query("
        SELECT 
            CASE 
                WHEN age BETWEEN 6 AND 10 THEN '6-10 Yaş'
                WHEN age BETWEEN 11 AND 15 THEN '11-15 Yaş'
                WHEN age BETWEEN 16 AND 20 THEN '16-20 Yaş'
                ELSE '20+ Yaş'
            END as age_group,
            COUNT(*) as count
        FROM athletes 
        WHERE status = 'active'
        GROUP BY age_group
        ORDER BY MIN(age)
    ")->fetchAll();
    
    // Aylık kayıt trendi
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(registration_date, '%Y-%m') as month,
            COUNT(*) as new_registrations
        FROM athletes 
        WHERE registration_date BETWEEN ? AND ?
        GROUP BY DATE_FORMAT(registration_date, '%Y-%m')
        ORDER BY month DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $report_data['registration_trend'] = $stmt->fetchAll();
    
} elseif ($report_type === 'payments') {
    // Aidat rapor verileri
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(payment_month, '%Y-%m') as month,
            COUNT(*) as total_payments,
            SUM(amount) as total_amount,
            SUM(CASE WHEN status = 'paid' THEN amount ELSE 0 END) as paid_amount,
            SUM(CASE WHEN status = 'pending' THEN amount ELSE 0 END) as pending_amount,
            SUM(CASE WHEN status = 'overdue' THEN amount ELSE 0 END) as overdue_amount
        FROM payments 
        WHERE payment_month BETWEEN ? AND ?
        GROUP BY DATE_FORMAT(payment_month, '%Y-%m')
        ORDER BY month DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $report_data['monthly_payments'] = $stmt->fetchAll();
    
    // Ödeme yöntemi analizi
    $stmt = $pdo->prepare("
        SELECT 
            payment_method,
            COUNT(*) as count,
            SUM(amount) as total_amount
        FROM payments 
        WHERE payment_month BETWEEN ? AND ? AND status = 'paid'
        GROUP BY payment_method
        ORDER BY total_amount DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $report_data['payment_methods'] = $stmt->fetchAll();
    
    // Geciken ödemeler
    $report_data['overdue_payments'] = $pdo->query("
        SELECT 
            p.*,
            a.first_name,
            a.last_name,
            DATEDIFF(NOW(), p.payment_month) as days_overdue
        FROM payments p
        JOIN athletes a ON p.athlete_id = a.id
        WHERE p.status = 'overdue'
        ORDER BY days_overdue DESC
    ")->fetchAll();
    
} elseif ($report_type === 'attendance') {
    // Yoklama rapor verileri
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(training_date, '%Y-%m') as month,
            COUNT(*) as total_attendance,
            SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) as present_count,
            SUM(CASE WHEN status = 'absent' THEN 1 ELSE 0 END) as absent_count,
            ROUND(SUM(CASE WHEN status = 'present' THEN 1 ELSE 0 END) * 100.0 / COUNT(*), 2) as attendance_rate
        FROM attendance 
        WHERE training_date BETWEEN ? AND ?
        GROUP BY DATE_FORMAT(training_date, '%Y-%m')
        ORDER BY month DESC
    ");
    $stmt->execute([$start_date, $end_date]);
    $report_data['monthly_attendance'] = $stmt->fetchAll();
    
    // Sporcu bazlı yoklama analizi
    $report_data['athlete_attendance'] = $pdo->query("
        SELECT 
            a.first_name,
            a.last_name,
            COUNT(att.id) as total_trainings,
            SUM(CASE WHEN att.status = 'present' THEN 1 ELSE 0 END) as present_count,
            ROUND(SUM(CASE WHEN att.status = 'present' THEN 1 ELSE 0 END) * 100.0 / COUNT(att.id), 2) as attendance_rate
        FROM athletes a
        LEFT JOIN attendance att ON a.id = att.athlete_id
        WHERE a.status = 'active'
        GROUP BY a.id, a.first_name, a.last_name
        HAVING total_trainings > 0
        ORDER BY attendance_rate DESC
    ")->fetchAll();
}

// PDF rapor oluşturma
if (isset($_GET['export']) && $_GET['export'] === 'pdf') {
    // Output buffer'ı temizle
    if (ob_get_level()) {
        ob_end_clean();
    }
    
    // HTML olarak PDF benzeri çıktı oluştur
    $html = generateReportHTML($report_type, $report_data, $start_date, $end_date);
    
    // Print-friendly CSS ekle
    $html = str_replace('</head>', '
    <style>
        @media print {
            body { margin: 0; padding: 15px; }
            .no-print { display: none !important; }
            table { page-break-inside: avoid; }
            .summary { page-break-inside: avoid; }
        }
        @page {
            margin: 1cm;
            size: A4;
        }
    </style>
    </head>', $html);
    
    // HTML olarak gönder (tarayıcıda açılır, Ctrl+P ile PDF yazdırılabilir)
    header('Content-Type: text/html; charset=utf-8');
    header('Content-Disposition: inline; filename="rapor_' . $report_type . '_' . date('Y-m-d_H-i-s') . '.html"');
    header('Cache-Control: no-cache, must-revalidate');
    header('Expires: Sat, 26 Jul 1997 05:00:00 GMT');
    
    // HTML'i gönder
    echo $html;
    exit;
}

// HTML rapor oluşturma
if (isset($_GET['export']) && $_GET['export'] === 'html') {
    // HTML olarak rapor oluştur
    $html = generateReportHTML($report_type, $report_data, $start_date, $end_date);
    
    // HTML header'ları
    header('Content-Type: text/html; charset=utf-8');
    header('Content-Disposition: attachment; filename="rapor_' . $report_type . '_' . date('Y-m-d_H-i-s') . '.html"');
    header('Cache-Control: no-cache, must-revalidate');
    header('Expires: Sat, 26 Jul 1997 05:00:00 GMT');
    
    // HTML'i gönder
    echo $html;
    exit;
}

// CSV rapor oluşturma
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    // Output buffer'ı temizle
    if (ob_get_level()) {
        ob_end_clean();
    }
    
    // CSV verilerini hazırla
    $csv_data = generateCSVData($report_type, $report_data, $start_date, $end_date);
    
    // CSV header'ları
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="rapor_' . $report_type . '_' . date('Y-m-d_H-i-s') . '.csv"');
    header('Cache-Control: no-cache, must-revalidate');
    header('Expires: Sat, 26 Jul 1997 05:00:00 GMT');
    
    // BOM ekle (Excel'de Türkçe karakterler için)
    echo "\xEF\xBB\xBF";
    
    // CSV verilerini gönder
    $output = fopen('php://output', 'w');
    foreach ($csv_data as $row) {
        fputcsv($output, $row);
    }
    fclose($output);
    exit;
}
?>

<div class="page-header">
    <div class="page-title">
        <i class="fas fa-chart-bar"></i>
        <h2>Raporlar ve Analiz</h2>
    </div>
    <div class="header-actions">
        <a href="?export=pdf&type=<?php echo $report_type; ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" class="btn btn-danger">
            <i class="fas fa-file-pdf"></i>
            <span>PDF İndir</span>
        </a>
        <a href="?export=html&type=<?php echo $report_type; ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" class="btn btn-secondary">
            <i class="fas fa-file-code"></i>
            <span>HTML İndir</span>
        </a>
        <a href="?export=csv&type=<?php echo $report_type; ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" class="btn btn-success">
            <i class="fas fa-file-csv"></i>
            <span>CSV İndir</span>
        </a>
    </div>
</div>

<!-- Rapor Filtreleri -->
<div class="report-filters">
    <div class="filter-tabs">
        <a href="?type=financial&period=<?php echo $period; ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" 
           class="filter-tab <?php echo $report_type === 'financial' ? 'active' : ''; ?>">
            <i class="fas fa-money-bill-wave"></i>
            <span>Mali Rapor</span>
        </a>
        <a href="?type=athletes&period=<?php echo $period; ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" 
           class="filter-tab <?php echo $report_type === 'athletes' ? 'active' : ''; ?>">
            <i class="fas fa-users"></i>
            <span>Sporcu Raporu</span>
        </a>
        <a href="?type=payments&period=<?php echo $period; ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" 
           class="filter-tab <?php echo $report_type === 'payments' ? 'active' : ''; ?>">
            <i class="fas fa-credit-card"></i>
            <span>Aidat Raporu</span>
        </a>
        <a href="?type=attendance&period=<?php echo $period; ?>&start_date=<?php echo $start_date; ?>&end_date=<?php echo $end_date; ?>" 
           class="filter-tab <?php echo $report_type === 'attendance' ? 'active' : ''; ?>">
            <i class="fas fa-clipboard-check"></i>
            <span>Yoklama Raporu</span>
        </a>
    </div>
    
    <div class="filter-controls">
        <form method="get" class="filter-form">
            <input type="hidden" name="type" value="<?php echo $report_type; ?>">
            <div class="form-row">
                <div class="form-group">
                    <label>Başlangıç Tarihi</label>
                    <input type="date" name="start_date" value="<?php echo $start_date; ?>" required>
                </div>
                <div class="form-group">
                    <label>Bitiş Tarihi</label>
                    <input type="date" name="end_date" value="<?php echo $end_date; ?>" required>
                </div>
                <div class="form-group">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-search"></i>
                        <span>Filtrele</span>
                    </button>
                </div>
            </div>
        </form>
    </div>
</div>

<?php if ($report_type === 'financial'): ?>
    <!-- Mali Rapor -->
    <div class="report-section">
        <div class="report-summary">
            <div class="summary-cards">
                <div class="summary-card income">
                    <div class="card-icon">
                        <i class="fas fa-arrow-up"></i>
                    </div>
                    <div class="card-content">
                        <h3><?php echo number_format($report_data['totals']['total_income'], 2); ?> ₺</h3>
                        <p>Toplam Gelir</p>
                        <small><?php echo $report_data['totals']['income_count']; ?> işlem</small>
                    </div>
                </div>
                
                <div class="summary-card expense">
                    <div class="card-icon">
                        <i class="fas fa-arrow-down"></i>
                    </div>
                    <div class="card-content">
                        <h3><?php echo number_format($report_data['totals']['total_expense'], 2); ?> ₺</h3>
                        <p>Toplam Gider</p>
                        <small><?php echo $report_data['totals']['expense_count']; ?> işlem</small>
                    </div>
                </div>
                
                <div class="summary-card balance <?php echo ($report_data['totals']['total_income'] - $report_data['totals']['total_expense']) >= 0 ? 'positive' : 'negative'; ?>">
                    <div class="card-icon">
                        <i class="fas fa-balance-scale"></i>
                    </div>
                    <div class="card-content">
                        <h3><?php echo number_format($report_data['totals']['total_income'] - $report_data['totals']['total_expense'], 2); ?> ₺</h3>
                        <p>Net Bakiye</p>
                        <small>Gelir - Gider</small>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Aylık Trend Grafiği -->
        <div class="chart-container">
            <h3><i class="fas fa-chart-line"></i> Aylık Mali Durum</h3>
            <div class="chart-content">
                <div class="trend-chart">
                    <?php foreach ($report_data['monthly'] as $month_data): ?>
                        <div class="trend-bar">
                            <div class="bar income" style="height: <?php echo max(20, ($month_data['income'] / max(1, max(array_column($report_data['monthly'], 'income')))) * 100); ?>%"></div>
                            <div class="bar expense" style="height: <?php echo max(20, ($month_data['expense'] / max(1, max(array_column($report_data['monthly'], 'expense')))) * 100); ?>%"></div>
                            <div class="bar-label"><?php echo date('M Y', strtotime($month_data['month'] . '-01')); ?></div>
                            <div class="bar-values">
                                <span class="income-value">+<?php echo number_format($month_data['income'], 0); ?>₺</span>
                                <span class="expense-value">-<?php echo number_format($month_data['expense'], 0); ?>₺</span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        
        <!-- Kategori Analizi -->
        <div class="table-container">
            <h3><i class="fas fa-tags"></i> Kategori Bazlı Analiz</h3>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Kategori</th>
                        <th>Tür</th>
                        <th>İşlem Sayısı</th>
                        <th>Toplam Tutar</th>
                        <th>Ortalama</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($report_data['categories'] as $category): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($category['category']); ?></td>
                            <td>
                                <span class="status status-<?php echo $category['transaction_type']; ?>">
                                    <?php echo $category['transaction_type'] === 'income' ? 'Gelir' : 'Gider'; ?>
                                </span>
                            </td>
                            <td><?php echo $category['transaction_count']; ?></td>
                            <td class="<?php echo $category['transaction_type'] === 'income' ? 'text-success' : 'text-danger'; ?>">
                                <?php echo number_format($category['total_amount'], 2); ?> ₺
                            </td>
                            <td><?php echo number_format($category['total_amount'] / $category['transaction_count'], 2); ?> ₺</td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>

<?php elseif ($report_type === 'athletes'): ?>
    <!-- Sporcu Raporu -->
    <div class="report-section">
        <div class="report-summary">
            <div class="summary-cards">
                <div class="summary-card">
                    <div class="card-icon">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="card-content">
                        <h3><?php echo $report_data['total_athletes']; ?></h3>
                        <p>Toplam Aktif Sporcu</p>
                    </div>
                </div>
                
                <div class="summary-card">
                    <div class="card-icon">
                        <i class="fas fa-user-plus"></i>
                    </div>
                    <div class="card-content">
                        <h3><?php echo $report_data['new_athletes']; ?></h3>
                        <p>Son 30 Gün Yeni Kayıt</p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Yaş Grupları -->
        <div class="chart-container">
            <h3><i class="fas fa-chart-pie"></i> Yaş Grupları Dağılımı</h3>
            <div class="chart-content">
                <div class="age-groups">
                    <?php foreach ($report_data['age_groups'] as $age_group): ?>
                        <div class="age-group-item">
                            <div class="age-group-info">
                                <span class="age-group-name"><?php echo $age_group['age_group']; ?></span>
                                <span class="age-group-count"><?php echo $age_group['count']; ?> sporcu</span>
                            </div>
                            <div class="age-group-bar">
                                <div class="age-group-progress" style="width: <?php echo ($age_group['count'] / max(1, $report_data['total_athletes'])) * 100; ?>%"></div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>

<?php elseif ($report_type === 'payments'): ?>
    <!-- Aidat Raporu -->
    <div class="report-section">
        <div class="table-container">
            <h3><i class="fas fa-credit-card"></i> Aylık Aidat Analizi</h3>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Ay</th>
                        <th>Toplam Aidat</th>
                        <th>Ödenen</th>
                        <th>Bekleyen</th>
                        <th>Geciken</th>
                        <th>Tahsilat Oranı</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($report_data['monthly_payments'] as $payment): ?>
                        <tr>
                            <td><?php echo date('M Y', strtotime($payment['month'] . '-01')); ?></td>
                            <td><?php echo number_format($payment['total_amount'], 2); ?> ₺</td>
                            <td class="text-success"><?php echo number_format($payment['paid_amount'], 2); ?> ₺</td>
                            <td class="text-warning"><?php echo number_format($payment['pending_amount'], 2); ?> ₺</td>
                            <td class="text-danger"><?php echo number_format($payment['overdue_amount'], 2); ?> ₺</td>
                            <td><?php echo round(($payment['paid_amount'] / max(1, $payment['total_amount'])) * 100, 1); ?>%</td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Geciken Ödemeler -->
        <?php if (!empty($report_data['overdue_payments'])): ?>
            <div class="table-container">
                <h3><i class="fas fa-exclamation-triangle"></i> Geciken Ödemeler</h3>
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Sporcu</th>
                            <th>Ay</th>
                            <th>Tutar</th>
                            <th>Gecikme Süresi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($report_data['overdue_payments'] as $overdue): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($overdue['first_name'] . ' ' . $overdue['last_name']); ?></td>
                                <td><?php echo date('M Y', strtotime($overdue['payment_month'] . '-01')); ?></td>
                                <td class="text-danger"><?php echo number_format($overdue['amount'], 2); ?> ₺</td>
                                <td class="text-danger"><?php echo $overdue['days_overdue']; ?> gün</td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>
    </div>

<?php elseif ($report_type === 'attendance'): ?>
    <!-- Yoklama Raporu -->
    <div class="report-section">
        <div class="table-container">
            <h3><i class="fas fa-clipboard-check"></i> Aylık Yoklama Analizi</h3>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Ay</th>
                        <th>Toplam Antrenman</th>
                        <th>Gelen</th>
                        <th>Gelmeyen</th>
                        <th>Katılım Oranı</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($report_data['monthly_attendance'] as $attendance): ?>
                        <tr>
                            <td><?php echo date('M Y', strtotime($attendance['month'] . '-01')); ?></td>
                            <td><?php echo $attendance['total_attendance']; ?></td>
                            <td class="text-success"><?php echo $attendance['present_count']; ?></td>
                            <td class="text-danger"><?php echo $attendance['absent_count']; ?></td>
                            <td>
                                <div class="progress-bar">
                                    <div class="progress-fill" style="width: <?php echo $attendance['attendance_rate']; ?>%"></div>
                                    <span class="progress-text"><?php echo $attendance['attendance_rate']; ?>%</span>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Sporcu Bazlı Yoklama -->
        <div class="table-container">
            <h3><i class="fas fa-user-check"></i> Sporcu Bazlı Katılım Oranları</h3>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Sporcu</th>
                        <th>Toplam Antrenman</th>
                        <th>Katıldığı</th>
                        <th>Katılım Oranı</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($report_data['athlete_attendance'] as $athlete): ?>
                        <tr>
                            <td><?php echo htmlspecialchars($athlete['first_name'] . ' ' . $athlete['last_name']); ?></td>
                            <td><?php echo $athlete['total_trainings']; ?></td>
                            <td class="text-success"><?php echo $athlete['present_count']; ?></td>
                            <td>
                                <div class="progress-bar">
                                    <div class="progress-fill" style="width: <?php echo $athlete['attendance_rate']; ?>%"></div>
                                    <span class="progress-text"><?php echo $athlete['attendance_rate']; ?>%</span>
                                </div>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
<?php endif; ?>

<?php
// PDF rapor HTML oluşturma fonksiyonu
function generateReportHTML($type, $data, $start_date, $end_date) {
    $html = '<!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>Rapor - ' . ucfirst($type) . '</title>
        <style>
            body { 
                font-family: Arial, sans-serif; 
                margin: 0; 
                padding: 20px; 
                background: white;
                color: #333;
            }
            .header { 
                text-align: center; 
                margin-bottom: 30px; 
                border-bottom: 2px solid #007bff;
                padding-bottom: 20px;
            }
            .header h1 { 
                color: #007bff; 
                margin: 0 0 10px 0; 
                font-size: 24px;
            }
            .header h2 { 
                color: #333; 
                margin: 0 0 10px 0; 
                font-size: 18px;
            }
            .header p { 
                color: #666; 
                margin: 0; 
                font-size: 14px;
            }
            .summary { 
                display: flex; 
                justify-content: space-around; 
                margin: 20px 0; 
                flex-wrap: wrap;
            }
            .summary-item { 
                text-align: center; 
                margin: 10px;
                padding: 15px;
                border: 1px solid #ddd;
                border-radius: 5px;
                min-width: 150px;
            }
            .summary-item h3 {
                margin: 0 0 5px 0;
                color: #007bff;
                font-size: 20px;
            }
            .summary-item p {
                margin: 0;
                color: #666;
                font-size: 14px;
            }
            table { 
                width: 100%; 
                border-collapse: collapse; 
                margin: 20px 0; 
                font-size: 12px;
            }
            th, td { 
                border: 1px solid #ddd; 
                padding: 8px; 
                text-align: left; 
            }
            th { 
                background-color: #f8f9fa; 
                font-weight: bold;
                color: #333;
            }
            .text-success { color: #28a745; font-weight: bold; }
            .text-danger { color: #dc3545; font-weight: bold; }
            .text-warning { color: #ffc107; font-weight: bold; }
            .section-title {
                color: #007bff;
                border-bottom: 1px solid #007bff;
                padding-bottom: 5px;
                margin: 30px 0 15px 0;
                font-size: 16px;
            }
            .footer {
                margin-top: 30px;
                text-align: center;
                color: #666;
                font-size: 12px;
                border-top: 1px solid #ddd;
                padding-top: 15px;
            }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>BAŞKENT 1910 GÜCÜ SK</h1>
            <h2>' . strtoupper($type) . ' RAPORU</h2>
            <p>Tarih Aralığı: ' . date('d.m.Y', strtotime($start_date)) . ' - ' . date('d.m.Y', strtotime($end_date)) . '</p>
            <p>Rapor Tarihi: ' . date('d.m.Y H:i') . '</p>
        </div>';
    
    // Rapor türüne göre içerik oluştur
    if ($type === 'financial') {
        $html .= generateFinancialReportContent($data);
    } elseif ($type === 'athletes') {
        $html .= generateAthletesReportContent($data);
    } elseif ($type === 'payments') {
        $html .= generatePaymentsReportContent($data);
    } elseif ($type === 'attendance') {
        $html .= generateAttendanceReportContent($data);
    }
    
    $html .= '
        <div class="footer">
            <p>Bu rapor BAŞKENT 1910 GÜCÜ SK yönetim sistemi tarafından otomatik olarak oluşturulmuştur.</p>
        </div>
    </body>
    </html>';
    
    return $html;
}

// Mali rapor içeriği
function generateFinancialReportContent($data) {
    $html = '<div class="section-title">MALİ DURUM ÖZETİ</div>';
    
    if (isset($data['totals'])) {
        $html .= '<div class="summary">
            <div class="summary-item">
                <h3>' . number_format($data['totals']['total_income'], 2) . ' ₺</h3>
                <p>Toplam Gelir</p>
            </div>
            <div class="summary-item">
                <h3>' . number_format($data['totals']['total_expense'], 2) . ' ₺</h3>
                <p>Toplam Gider</p>
            </div>
            <div class="summary-item">
                <h3>' . number_format($data['totals']['total_income'] - $data['totals']['total_expense'], 2) . ' ₺</h3>
                <p>Net Bakiye</p>
            </div>
        </div>';
    }
    
    if (isset($data['categories']) && !empty($data['categories'])) {
        $html .= '<div class="section-title">KATEGORİ BAZLI ANALİZ</div>
        <table>
            <thead>
                <tr>
                    <th>Kategori</th>
                    <th>Tür</th>
                    <th>İşlem Sayısı</th>
                    <th>Toplam Tutar</th>
                </tr>
            </thead>
            <tbody>';
        
        foreach ($data['categories'] as $category) {
            $html .= '<tr>
                <td>' . htmlspecialchars($category['category']) . '</td>
                <td>' . ($category['transaction_type'] === 'income' ? 'Gelir' : 'Gider') . '</td>
                <td>' . $category['transaction_count'] . '</td>
                <td class="' . ($category['transaction_type'] === 'income' ? 'text-success' : 'text-danger') . '">' . number_format($category['total_amount'], 2) . ' ₺</td>
            </tr>';
        }
        
        $html .= '</tbody></table>';
    }
    
    return $html;
}

// Sporcu rapor içeriği
function generateAthletesReportContent($data) {
    $html = '<div class="section-title">SPORCU İSTATİSTİKLERİ</div>';
    
    $html .= '<div class="summary">
        <div class="summary-item">
            <h3>' . $data['total_athletes'] . '</h3>
            <p>Toplam Aktif Sporcu</p>
        </div>
        <div class="summary-item">
            <h3>' . $data['new_athletes'] . '</h3>
            <p>Son 30 Gün Yeni Kayıt</p>
        </div>
    </div>';
    
    if (isset($data['age_groups']) && !empty($data['age_groups'])) {
        $html .= '<div class="section-title">YAŞ GRUPLARI DAĞILIMI</div>
        <table>
            <thead>
                <tr>
                    <th>Yaş Grubu</th>
                    <th>Sporcu Sayısı</th>
                    <th>Oran</th>
                </tr>
            </thead>
            <tbody>';
        
        foreach ($data['age_groups'] as $age_group) {
            $percentage = round(($age_group['count'] / max(1, $data['total_athletes'])) * 100, 1);
            $html .= '<tr>
                <td>' . htmlspecialchars($age_group['age_group']) . '</td>
                <td>' . $age_group['count'] . '</td>
                <td>' . $percentage . '%</td>
            </tr>';
        }
        
        $html .= '</tbody></table>';
    }
    
    return $html;
}

// Aidat rapor içeriği
function generatePaymentsReportContent($data) {
    $html = '<div class="section-title">AİDAT ANALİZİ</div>';
    
    if (isset($data['monthly_payments']) && !empty($data['monthly_payments'])) {
        $html .= '<table>
            <thead>
                <tr>
                    <th>Ay</th>
                    <th>Toplam Aidat</th>
                    <th>Ödenen</th>
                    <th>Bekleyen</th>
                    <th>Geciken</th>
                    <th>Tahsilat Oranı</th>
                </tr>
            </thead>
            <tbody>';
        
        foreach ($data['monthly_payments'] as $payment) {
            $collection_rate = round(($payment['paid_amount'] / max(1, $payment['total_amount'])) * 100, 1);
            $html .= '<tr>
                <td>' . date('M Y', strtotime($payment['month'] . '-01')) . '</td>
                <td>' . number_format($payment['total_amount'], 2) . ' ₺</td>
                <td class="text-success">' . number_format($payment['paid_amount'], 2) . ' ₺</td>
                <td class="text-warning">' . number_format($payment['pending_amount'], 2) . ' ₺</td>
                <td class="text-danger">' . number_format($payment['overdue_amount'], 2) . ' ₺</td>
                <td>' . $collection_rate . '%</td>
            </tr>';
        }
        
        $html .= '</tbody></table>';
    }
    
    return $html;
}

// Yoklama rapor içeriği
function generateAttendanceReportContent($data) {
    $html = '<div class="section-title">YOKLAMA ANALİZİ</div>';
    
    if (isset($data['monthly_attendance']) && !empty($data['monthly_attendance'])) {
        $html .= '<table>
            <thead>
                <tr>
                    <th>Ay</th>
                    <th>Toplam Antrenman</th>
                    <th>Gelen</th>
                    <th>Gelmeyen</th>
                    <th>Katılım Oranı</th>
                </tr>
            </thead>
            <tbody>';
        
        foreach ($data['monthly_attendance'] as $attendance) {
            $html .= '<tr>
                <td>' . date('M Y', strtotime($attendance['month'] . '-01')) . '</td>
                <td>' . $attendance['total_attendance'] . '</td>
                <td class="text-success">' . $attendance['present_count'] . '</td>
                <td class="text-danger">' . $attendance['absent_count'] . '</td>
                <td>' . $attendance['attendance_rate'] . '%</td>
            </tr>';
        }
        
        $html .= '</tbody></table>';
    }
    
    return $html;
}

// CSV veri oluşturma fonksiyonu
function generateCSVData($type, $data, $start_date, $end_date) {
    $csv_data = [];
    
    // Başlık satırı
    $csv_data[] = ['BAŞKENT 1910 GÜCÜ SK - ' . strtoupper($type) . ' RAPORU'];
    $csv_data[] = ['Tarih Aralığı: ' . date('d.m.Y', strtotime($start_date)) . ' - ' . date('d.m.Y', strtotime($end_date))];
    $csv_data[] = ['Rapor Tarihi: ' . date('d.m.Y H:i')];
    $csv_data[] = []; // Boş satır
    
    if ($type === 'financial') {
        if (isset($data['totals'])) {
            $csv_data[] = ['MALİ DURUM ÖZETİ'];
            $csv_data[] = ['Toplam Gelir', number_format($data['totals']['total_income'], 2) . ' ₺'];
            $csv_data[] = ['Toplam Gider', number_format($data['totals']['total_expense'], 2) . ' ₺'];
            $csv_data[] = ['Net Bakiye', number_format($data['totals']['total_income'] - $data['totals']['total_expense'], 2) . ' ₺'];
            $csv_data[] = []; // Boş satır
        }
        
        if (isset($data['categories']) && !empty($data['categories'])) {
            $csv_data[] = ['KATEGORİ BAZLI ANALİZ'];
            $csv_data[] = ['Kategori', 'Tür', 'İşlem Sayısı', 'Toplam Tutar'];
            foreach ($data['categories'] as $category) {
                $csv_data[] = [
                    $category['category'],
                    $category['transaction_type'] === 'income' ? 'Gelir' : 'Gider',
                    $category['transaction_count'],
                    number_format($category['total_amount'], 2) . ' ₺'
                ];
            }
        }
        
    } elseif ($type === 'athletes') {
        $csv_data[] = ['SPORCU İSTATİSTİKLERİ'];
        $csv_data[] = ['Toplam Aktif Sporcu', $data['total_athletes']];
        $csv_data[] = ['Son 30 Gün Yeni Kayıt', $data['new_athletes']];
        $csv_data[] = []; // Boş satır
        
        if (isset($data['age_groups']) && !empty($data['age_groups'])) {
            $csv_data[] = ['YAŞ GRUPLARI DAĞILIMI'];
            $csv_data[] = ['Yaş Grubu', 'Sporcu Sayısı', 'Oran (%)'];
            foreach ($data['age_groups'] as $age_group) {
                $percentage = round(($age_group['count'] / max(1, $data['total_athletes'])) * 100, 1);
                $csv_data[] = [
                    $age_group['age_group'],
                    $age_group['count'],
                    $percentage . '%'
                ];
            }
        }
        
    } elseif ($type === 'payments') {
        if (isset($data['monthly_payments']) && !empty($data['monthly_payments'])) {
            $csv_data[] = ['AİDAT ANALİZİ'];
            $csv_data[] = ['Ay', 'Toplam Aidat', 'Ödenen', 'Bekleyen', 'Geciken', 'Tahsilat Oranı (%)'];
            foreach ($data['monthly_payments'] as $payment) {
                $collection_rate = round(($payment['paid_amount'] / max(1, $payment['total_amount'])) * 100, 1);
                $csv_data[] = [
                    date('M Y', strtotime($payment['month'] . '-01')),
                    number_format($payment['total_amount'], 2) . ' ₺',
                    number_format($payment['paid_amount'], 2) . ' ₺',
                    number_format($payment['pending_amount'], 2) . ' ₺',
                    number_format($payment['overdue_amount'], 2) . ' ₺',
                    $collection_rate . '%'
                ];
            }
        }
        
    } elseif ($type === 'attendance') {
        if (isset($data['monthly_attendance']) && !empty($data['monthly_attendance'])) {
            $csv_data[] = ['YOKLAMA ANALİZİ'];
            $csv_data[] = ['Ay', 'Toplam Antrenman', 'Gelen', 'Gelmeyen', 'Katılım Oranı (%)'];
            foreach ($data['monthly_attendance'] as $attendance) {
                $csv_data[] = [
                    date('M Y', strtotime($attendance['month'] . '-01')),
                    $attendance['total_attendance'],
                    $attendance['present_count'],
                    $attendance['absent_count'],
                    $attendance['attendance_rate'] . '%'
                ];
            }
        }
    }
    
    return $csv_data;
}
?>

<?php require_once __DIR__ . '/../includes/footer.php'; ?>
