<?php
// Genel fonksiyonlar

/**
 * Veritabanı bağlantısı oluştur
 */
function getDBConnection() {
    static $pdo = null;
    
    if ($pdo === null) {
        try {
            $host = 'localhost';
            $dbname = 'baskent1910guksk';
            $username = 'root';
            $password = '';
            
            $dsn = "mysql:host=$host;dbname=$dbname;charset=utf8mb4";
            $pdo = new PDO($dsn, $username, $password, [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ]);
        } catch (PDOException $e) {
            die('Veritabanı bağlantı hatası: ' . $e->getMessage());
        }
    }
    
    return $pdo;
}

/**
 * Güvenli HTML çıktısı
 */
function h($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

/**
 * Tarih formatla
 */
function formatDate($date, $format = 'd.m.Y H:i') {
    return date($format, strtotime($date));
}

/**
 * Kısa metin oluştur
 */
function truncateText($text, $length = 100) {
    if (strlen($text) <= $length) {
        return $text;
    }
    return substr($text, 0, $length) . '...';
}

/**
 * Başarı mesajı
 */
function setSuccessMessage($message) {
    $_SESSION['success_message'] = $message;
}

/**
 * Hata mesajı
 */
function setErrorMessage($message) {
    $_SESSION['error_message'] = $message;
}

/**
 * Mesajları göster ve temizle
 */
function showMessages() {
    if (isset($_SESSION['success_message'])) {
        echo '<div class="alert alert-success">' . h($_SESSION['success_message']) . '</div>';
        unset($_SESSION['success_message']);
    }
    
    if (isset($_SESSION['error_message'])) {
        echo '<div class="alert alert-danger">' . h($_SESSION['error_message']) . '</div>';
        unset($_SESSION['error_message']);
    }
}

/**
 * CSRF token oluştur
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * CSRF token doğrula
 */
function validateCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Kullanıcı giriş kontrolü
 */
function checkLogin() {
    if (!isset($_SESSION['user_id'])) {
        header('Location: /admin/login.php');
        exit();
    }
}

/**
 * Admin yetki kontrolü
 */
function checkAdmin() {
    checkLogin();
    if (!isset($_SESSION['is_admin']) || !$_SESSION['is_admin']) {
        header('Location: /admin/');
        exit();
    }
}

/**
 * Sayfa başlığı oluştur
 */
function getPageTitle($title) {
    return $title . ' - BAŞKENT 1910 GÜCÜ SK';
}

/**
 * URL oluştur
 */
function url($path = '') {
    $base = '/admin/';
    return $base . ltrim($path, '/');
}

/**
 * Dosya boyutu formatla
 */
function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    
    $bytes /= (1 << (10 * $pow));
    
    return round($bytes, 2) . ' ' . $units[$pow];
}

/**
 * Rastgele string oluştur
 */
function generateRandomString($length = 10) {
    return substr(str_shuffle(str_repeat($x='0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ', ceil($length/strlen($x)) )),1,$length);
}

/**
 * Email doğrula
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Telefon doğrula
 */
function validatePhone($phone) {
    $phone = preg_replace('/[^0-9]/', '', $phone);
    return strlen($phone) >= 10 && strlen($phone) <= 15;
}

/**
 * Güçlü şifre kontrolü
 */
function validatePassword($password) {
    return strlen($password) >= 8 && 
           preg_match('/[A-Z]/', $password) && 
           preg_match('/[a-z]/', $password) && 
           preg_match('/[0-9]/', $password);
}

/**
 * JSON yanıt gönder
 */
function jsonResponse($data = null, $success = true, $message = '', $status = 200) {
    http_response_code($status);
    header('Content-Type: application/json');
    
    $response = [
        'success' => $success,
        'message' => $message,
        'data' => $data,
        'timestamp' => date('Y-m-d H:i:s')
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE);
    exit();
}

/**
 * Log yaz
 */
function writeLog($message, $level = 'INFO') {
    $logFile = __DIR__ . '/../logs/app.log';
    $logDir = dirname($logFile);
    
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[$timestamp] [$level] $message" . PHP_EOL;
    
    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
}

/**
 * Resim yükleme fonksiyonu
 */
function uploadImage($file, $folder = 'lesson_notes') {
    // İzin verilen dosya türleri
    $allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
    
    // Dosya kontrolü
    if (!isset($file['tmp_name']) || !is_uploaded_file($file['tmp_name'])) {
        return ['success' => false, 'message' => 'Dosya yüklenemedi'];
    }
    
    // Dosya türü kontrolü
    $fileType = mime_content_type($file['tmp_name']);
    if (!in_array($fileType, $allowedTypes)) {
        return ['success' => false, 'message' => 'Sadece JPG, PNG, GIF ve WebP dosyaları yüklenebilir'];
    }
    
    // Dosya boyutu kontrolü (5MB)
    if ($file['size'] > 5 * 1024 * 1024) {
        return ['success' => false, 'message' => 'Dosya boyutu 5MB\'dan büyük olamaz'];
    }
    
    // Klasör oluştur
    $uploadDir = __DIR__ . '/../uploads/' . $folder . '/';
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    // Dosya adı oluştur
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $fileName = uniqid() . '_' . time() . '.' . $extension;
    $filePath = $uploadDir . $fileName;
    
    // Dosyayı taşı
    if (move_uploaded_file($file['tmp_name'], $filePath)) {
        // Web erişilebilir yol
        $webPath = '/admin/uploads/' . $folder . '/' . $fileName;
        
        return [
            'success' => true,
            'file_path' => $filePath,
            'web_path' => $webPath,
            'file_name' => $fileName
        ];
    } else {
        return ['success' => false, 'message' => 'Dosya kaydedilemedi'];
    }
}

/**
 * Resim silme fonksiyonu
 */
function deleteImage($imagePath) {
    if (empty($imagePath)) {
        return true;
    }
    
    // Web path'den dosya path'ine çevir
    $filePath = str_replace('/admin/uploads/', __DIR__ . '/../uploads/', $imagePath);
    
    if (file_exists($filePath)) {
        return unlink($filePath);
    }
    
    return true;
}

/**
 * Resim boyutlandırma
 */
function resizeImage($sourcePath, $destinationPath, $maxWidth = 800, $maxHeight = 600) {
    $imageInfo = getimagesize($sourcePath);
    if (!$imageInfo) {
        return false;
    }
    
    $originalWidth = $imageInfo[0];
    $originalHeight = $imageInfo[1];
    $mimeType = $imageInfo['mime'];
    
    // Oranları hesapla
    $ratio = min($maxWidth / $originalWidth, $maxHeight / $originalHeight);
    $newWidth = intval($originalWidth * $ratio);
    $newHeight = intval($originalHeight * $ratio);
    
    // Kaynak resmi oluştur
    switch ($mimeType) {
        case 'image/jpeg':
            $sourceImage = imagecreatefromjpeg($sourcePath);
            break;
        case 'image/png':
            $sourceImage = imagecreatefrompng($sourcePath);
            break;
        case 'image/gif':
            $sourceImage = imagecreatefromgif($sourcePath);
            break;
        case 'image/webp':
            $sourceImage = imagecreatefromwebp($sourcePath);
            break;
        default:
            return false;
    }
    
    if (!$sourceImage) {
        return false;
    }
    
    // Yeni resmi oluştur
    $newImage = imagecreatetruecolor($newWidth, $newHeight);
    
    // PNG ve GIF için şeffaflık koru
    if ($mimeType === 'image/png' || $mimeType === 'image/gif') {
        imagealphablending($newImage, false);
        imagesavealpha($newImage, true);
        $transparent = imagecolorallocatealpha($newImage, 255, 255, 255, 127);
        imagefilledrectangle($newImage, 0, 0, $newWidth, $newHeight, $transparent);
    }
    
    // Resmi boyutlandır
    imagecopyresampled($newImage, $sourceImage, 0, 0, 0, 0, $newWidth, $newHeight, $originalWidth, $originalHeight);
    
    // Resmi kaydet
    $result = false;
    switch ($mimeType) {
        case 'image/jpeg':
            $result = imagejpeg($newImage, $destinationPath, 85);
            break;
        case 'image/png':
            $result = imagepng($newImage, $destinationPath, 8);
            break;
        case 'image/gif':
            $result = imagegif($newImage, $destinationPath);
            break;
        case 'image/webp':
            $result = imagewebp($newImage, $destinationPath, 85);
            break;
    }
    
    // Belleği temizle
    imagedestroy($sourceImage);
    imagedestroy($newImage);
    
    return $result;
}
?>
